//----------------------------------------------------------------------------
//
// Copyright (C) Sartorius Stedim Data Analytics AB 2017 -
//
// Use, modification and distribution are subject to the Boost Software
// License, Version 1.0. (See http://www.boost.org/LICENSE_1_0.txt)
//
//----------------------------------------------------------------------------

#ifndef _EZQ_H_
#define _EZQ_H_

//////////////////////////////////////////////////////////////////////////
// Includes
#include "SIMCAQP.h"
#include <vector>
#include <string>

#ifndef NULL
#define NULL   0
#endif

#ifndef BOOL
#define BOOL   int
#endif

#ifndef TRUE
#define TRUE   1
#endif

#ifndef FALSE
#define FALSE  0
#endif

//////////////////////////////////////////////////////////////////////////
// Helper classes to auto close containers then the object goes out of scope

// Class to encapsulate the Float matrix

class CEzFloatMatrix
{
public:
   CEzFloatMatrix() = default;
   CEzFloatMatrix(SQ_FloatMatrix pMat, bool bDeleteIt) : m_matrix(pMat), mbClearMatrix(bDeleteIt) { }
   ~CEzFloatMatrix()
   {
      if (mbClearMatrix)
         SQ_ClearFloatMatrix(&m_matrix);
   }
   //when a SQ_VectorData owns the matrix should not be deleted.
   void DoNotClearMatrix() { mbClearMatrix = false; }

   SQ_FloatMatrix* get() { return &m_matrix; }
   const SQ_FloatMatrix* get() const { return &m_matrix; }
   operator const SQ_FloatMatrix& () const { return m_matrix; }

   float operator()(int iRow, int iCol) const
   {
      float fVal;
      SQ_GetDataFromFloatMatrix(m_matrix, iRow, iCol, &fVal);
      return fVal;
   }

   int SizeRows() const
   {
      int iNumRows;
      SQ_GetNumRowsInFloatMatrix(m_matrix, &iNumRows);
      return iNumRows;
   }
   int SizeColumns() const
   {
      int iNumCols;
      SQ_GetNumColumnsInFloatMatrix(m_matrix, &iNumCols);
      return iNumCols;
   }

protected:
   CEzFloatMatrix(CEzFloatMatrix&) = delete;
   CEzFloatMatrix& operator=(const CEzFloatMatrix&) = delete;
   SQ_FloatMatrix m_matrix = nullptr;
   bool mbClearMatrix = true;
};

class CEzFloatVector
{
public:
   CEzFloatVector() = default;
   explicit CEzFloatVector(SQ_FloatVector pData) : m_data(pData) {}
   explicit CEzFloatVector(int iSize)
   {
      SQ_InitFloatVector(&m_data, iSize);
   }
   ~CEzFloatVector()
   {
      SQ_ClearFloatVector(&m_data);
   }

   SQ_FloatVector* get() { return &m_data; }
   const SQ_FloatVector* get() const { return &m_data; }
   operator const SQ_FloatVector& () const { return m_data; }

   float operator[](int iPos) const
   {
      float fVal;
      SQ_GetDataFromFloatVector(m_data, iPos, &fVal);
      return fVal;
   }
   void operator()(int iPos, float fVal) const
   {
      SQ_SetDataInFloatVector(m_data, iPos, fVal);
   }

   int size() const
   {
      int iSize;
      SQ_GetFloatVectorSize(m_data, &iSize);
      return iSize;
   }
protected:
   CEzFloatVector(CEzFloatVector&) = delete;
   CEzFloatVector& operator=(const CEzFloatVector&) = delete;
   SQ_FloatVector m_data = nullptr;
};

// Class to encapsulate the String vector
class CEzStringVector
{
public:
   CEzStringVector() = default;
   explicit CEzStringVector(int iSize)
   {
      SQ_InitStringVector(&m_vector, iSize);
   }
   ~CEzStringVector() { if (m_vector) SQ_ClearStringVector(&m_vector); }

   SQ_StringVector* get() { return &m_vector; }
   const SQ_StringVector* get() const { return &m_vector; }
   operator SQ_StringVector& () { return m_vector; }

   int size() const
   {
      int iSize = 0;
      SQ_GetNumStringsInVector(m_vector, &iSize);
      return iSize;
   }
   void operator()(int iPos, const std::string& strVal) const
   {
      SQ_SetStringInVector(m_vector, iPos, strVal.c_str());
   }

   std::string operator[](int iPos) const
   {
      char strVal[1000];
      if (SQ_GetStringFromVector(m_vector, iPos, strVal, sizeof(strVal)) == SQ_E_OK)
         return strVal;
      return "";
   }

private:
   CEzStringVector(const CEzStringVector&) = delete;
   CEzStringVector& operator=(const CEzStringVector&) = delete;
   SQ_StringVector m_vector = nullptr;
};

// Class to encapsulate Vector data
class CEzVectorData
{
public:
   CEzVectorData() = default;
   ~CEzVectorData() { if (m_vectorData) SQ_ClearVectorData(&m_vectorData); }

   void GetMatrix(CEzFloatMatrix& rMat)
   {
      SQ_GetDataMatrix(m_vectorData, rMat.get());
      rMat.DoNotClearMatrix();
   }

   void GetRowNames(CEzStringVector& rVec) { SQ_GetRowNames(m_vectorData, rVec.get()); }

   void GetColumnNames(CEzStringVector& rVec) { SQ_GetColumnNames(m_vectorData, rVec.get()); }

   SQ_VectorData* get() { return &m_vectorData; }
   const SQ_VectorData* get() const { return &m_vectorData; }
   operator SQ_VectorData& () { return m_vectorData; }

private:
   CEzVectorData(CEzVectorData&) = delete;
   CEzVectorData& operator=(const CEzVectorData&) = delete;
   SQ_VectorData m_vectorData = nullptr;
};


// Class to encapsulate the Int vector
class CEzIntVector
{
public:
   CEzIntVector() = default;
   explicit CEzIntVector(int iSize)
   {
      SQ_InitIntVector(&m_vector, iSize);
   }
   ~CEzIntVector() { if (m_vector) SQ_ClearIntVector(&m_vector); }

   SQ_IntVector* get() { return &m_vector; }
   const SQ_IntVector* get() const { return &m_vector; }
   operator SQ_IntVector& () { return m_vector; }

   int size() const
   {
      int iSize = 0;
      SQ_GetIntVectorSize(m_vector, &iSize);
      return iSize;
   }
   void operator()(int iPos, int iNewVal) const
   {
      SQ_SetDataInIntVector(m_vector, iPos, iNewVal);
   }
   int operator[](int iPos) const
   {
      int iVal = 0;
      SQ_GetDataFromIntVector(m_vector, iPos, &iVal);
      return iVal;
   }
private:
   CEzIntVector(CEzIntVector&) = delete;
   CEzIntVector& operator=(const CEzIntVector&) = delete;
   SQ_IntVector m_vector = nullptr;
};

//////////////////////////////////////////////////////////////////////////
// The EzQ class to easily do a prediction
//
// Example of how to use it:
// CEzQ oEzQ;
// oEzQ.OpenProject(PathToProject.usp);
// oEzQ.SetModel(2);
// oEzQ.Predict(myQuantitativeData, quantRows, quantCols, myQualitativeData, qualRows, qualCols);
// oEzQ.GetTPs(resultMatrix);
//
class CEzQ
{
public:
   CEzQ();
   ~CEzQ();

   /// Open the project.
   /// @param szUSPFile    In: is the full path to the file.
   /// @param szPassword   In: The password of the file. Can be NULL if not used.
   /// @return             Returns non zero if no error.
   SQ_ErrorCode OpenProject(const char* szUSPFile, const char* szPassword);

   /// Close the project. This is also done automatically by the 
   /// destructor when the object goes out of scope.
   /// @return             Returns non zero if no error.
   SQ_ErrorCode CloseProject();

   /// Set the index of the model to use. By default the first model is used.
   /// @param iModelIndex  In: The index of the model starting from 1 
   /// @return             Returns non zero if no error.
   SQ_ErrorCode SetModel(int iModelIndex);

   /// Set the data of the prediction set to use. Checks variable names so that we know that the correct predictions are made.
   /// @param quantitativeMatrix  In: A matrix with quantitative prediction data, row by row
   /// @param quantRowSize            In: Number of rows in quantitative prediction data
   /// @param quantColSize            In: Number of columns in the quantitative prediction data
   /// @param columnNames             In: vector with the names of the columns
   /// @return             Returns non zero if no error.
   SQ_ErrorCode Predict(const float* fMatQuantitativeData, int nQuantRows, int nQuantCols, const char* const* szColumnNames);

   /// Set the data of the prediction set to use. Assumes all data is in the correct order already, no name checking.
   /// @param fMatQuantitativeData  In: A matrix with quantitative prediction data, row by row
   /// @param nQuantRows            In: Number of rows in quantitative prediction data
   /// @param nQuantCols            In: Number of columns in the quantitative prediction data
   /// @param szMatQualitativeData  In: A matrix with qualitative prediction data, row by row
   /// @param nQualRows             In: Number of rows in qualitative prediction data
   /// @param nQuantCols            In: Number of columns in the qualitative prediction data
   /// @return             Returns non zero if no error.
   SQ_ErrorCode Predict2(const float* fMatQuantitativeData, int nQuantRows, int nQuantCols, const char* const* szMatQualitativeData, int nQuanlRows, int nQuanlCols);

   /// Get predicted T (scores)
   /// @param fmatResult         Out: The predicted score vector
   /// @return             Returns non zero if no error.
   SQ_ErrorCode GetTPS(CEzVectorData& fmatResult);

   /// Get predicted DModX
   /// @param fmatResult         Out: The predicted DModX vector
   /// @return             Returns non zero if no error.
   SQ_ErrorCode GetDModXPS(CEzVectorData& fmatResult);

   /// Get predicted DModY
   /// @param fmatResult         Out: The predicted DModY vector
   /// @return             Returns non zero if no error.
   SQ_ErrorCode GetDModYPS(CEzVectorData& fmatResult);

   /// Get predicted Y
   /// @param fmatResult         Out: The predicted Y vector
   /// @return             Returns non zero if no error.
   SQ_ErrorCode GetYPredPS(CEzVectorData& fmatResult);

   /// Get predicted T2
   /// @param fmatResult         Out: The predicted T2 vector
   /// @return             Returns non zero if no error.
   SQ_ErrorCode GetT2RangePS(CEzVectorData& fmatResult);

   /// Get the model T2Range critical limit
   /// @param fmatResult         Out: The T2Range critical limit of the model
   /// @return             Returns non zero if no error.
   SQ_ErrorCode GetT2RangeCrit(float& fResult);

   /// Returns the project handle to use some of the more advanced features of SIMCA-Q
   /// @return             Project handle
   SQ_Project GetProjectHandle() { return m_hProject; }

   /// Returns the prediction handle to use some of the more advanced features of SIMCA-Q
   /// @return             Prediction handle
   SQ_Prediction GetPredictionHandle() { return m_hPrediction; }

   std::string GetErrorDescription(SQ_ErrorCode eError) const;

protected:
   SQ_Project              m_hProject = nullptr;      /// The project handle, valid after the project is opened
   SQ_Prediction           m_hPrediction = nullptr;   /// The prediction handle, valid after a prediction is made
   CEzIntVector	         m_oComponents;             /// An array of components
   SQ_Model                m_hModel = nullptr;        /// The project handle, valid after the project is opened
   int                     m_iComponents = -1;        /// Total number of components
   float                   m_fProbLevel = -1.0f;      /// Model probability level
   float                   m_fDModXCrit = -1.0f;;     /// DModX critical limit
   float                   m_fHotT2Crit = -1.0f;;     /// Hotellings T2 critical limit
};

#endif // _EZQ_H_
